import React from 'react';
import Navbar from '@/components/Navbar';
import Footer from '@/components/Footer';
import dbConnect from '@/lib/db';
import Post from '@/models/Post';
import { notFound } from 'next/navigation';
import { Calendar, User, Tag } from 'lucide-react';
import Link from 'next/link';

export const revalidate = 60;

async function getPost(slug: string) {
  await dbConnect();
  const post = await (Post as any).findOne({ slug, status: 'published' });
  if (!post) return null;
  return JSON.parse(JSON.stringify(post));
}

export default async function SinglePostPage(props: { params: Promise<{ slug: string }> }) {
  const params = await props.params;
  const post = await getPost(params.slug);

  if (!post) return notFound();

  return (
    <div className="min-h-screen flex flex-col">
      <Navbar />
      
      <main className="flex-grow bg-white dark:bg-slate-900">
        {/* Header */}
        <div className="bg-slate-900 text-white py-20 relative overflow-hidden">
          {post.featuredImage && (
            <div className="absolute inset-0 z-0">
               <img src={post.featuredImage} alt="" className="w-full h-full object-cover opacity-20" />
               <div className="absolute inset-0 bg-gradient-to-t from-slate-900 to-transparent"></div>
            </div>
          )}
          <div className="max-w-4xl mx-auto px-4 relative z-10 text-center">
            <div className="flex items-center justify-center gap-4 text-sm text-slate-300 mb-4">
              <span className="flex items-center gap-1"><Calendar size={14} /> {new Date(post.createdAt).toLocaleDateString()}</span>
              {post.categories && post.categories.length > 0 && (
                <span className="flex items-center gap-1">
                  <Tag size={14} /> 
                  {post.categories.map((c: string) => <span key={c} className="capitalize">{c}</span>)}
                </span>
              )}
            </div>
            <h1 className="text-3xl md:text-5xl font-bold font-display leading-tight mb-6">{post.title}</h1>
          </div>
        </div>

        {/* Content */}
        <div className="max-w-4xl mx-auto px-4 py-16">
          {/* Main Body */}
          <div 
            className="prose prose-lg dark:prose-invert max-w-none mb-12"
            dangerouslySetInnerHTML={{ __html: post.content }}
          />

          {/* Gallery */}
          {post.gallery && post.gallery.length > 0 && (
            <div className="mt-16 pt-16 border-t border-slate-200 dark:border-slate-800">
              <h3 className="text-2xl font-bold mb-8 text-slate-900 dark:text-white">Image Gallery</h3>
              <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
                {post.gallery.map((img: string, idx: number) => (
                  <div key={idx} className="rounded-lg overflow-hidden shadow-sm hover:shadow-md transition aspect-square">
                    <img src={img} alt={`Gallery ${idx + 1}`} className="w-full h-full object-cover hover:scale-105 transition duration-500" />
                  </div>
                ))}
              </div>
            </div>
          )}
        </div>
      </main>

      <Footer />
    </div>
  );
}